#
# apacheModuleManager.tcl --
#
#  The apacheModuleManager manages information about
# different aspects of apache modules, as described in apacheModuleFormat.txt
# in the docs directory.
#  The apacheModuleManager has one propertyPagesManager per module, and keeps
# track internally of which property pages belong to which modules.
#   Things that apacheModuleManager allows:
#   
#   - Enable/disable modules on the fly
#   - Request all property pages associated with a certain node
#     (i.e virtual host) and a certain skill level (beginner, advanced)

class apacheModuleManager {
    variable nodeArray
    variable moduleArray
    variable icon
    variable description
    variable directiveModuleRelationship
    constructor {} {

	#  This array is going to be indexed by nodename, module
	#   nodeArray($module,$nodeName)
	#                             
	#  Each one of the elements is a triplet:
	#     pageName skillLevel hookPoint   

        array set nodeArray ""
	array set icon ""
	array set description ""
	
	# Module list. The value of the array should be 1 or 0 
	# (enabled/disabled)

        array set moduleArray ""
	array set directiveModuleRelationship ""
    }
    method addModuleDescription {name desc}
    method addModuleIcon {name moduleIcon}
    method getModuleDescription {name}
    method getModuleIcon {name}
    method getPropertyPagesByNodetypeList { nodeType }
    method addPropertyPageNodeRelationship { module pageName nodeType args}
    method getEnabledModuleList {}
    method getDisabledModuleList {}    
    method disableModule {module}
    method enableModule {module}
    method addModuleDirectivesRelationship 
    method isDirectiveEnabled
}
    
body apacheModuleManager::addModuleDescription {module desc} {
    set description($module) $desc
}

body apacheModuleManager::addModuleIcon {module moduleIcon} {
    set icon($module) $moduleIcon
}

body apacheModuleManager::getModuleDescription {module} {
    return $description($module)
}

body apacheModuleManager::getModuleIcon {module} {
     return $icon($module)
}

# getEnabledModules --
#     Get all modules currently enabled
#

body apacheModuleManager::getEnabledModuleList {} {
    set result ""
    foreach one [array names moduleArray] {
	if $moduleArray($one) {
   	    lappend result $one
	}
    }	
    return $result 
}

# getDisabledModules --
#     Get all modules currently disabled
#

body apacheModuleManager::getDisabledModuleList {} {
     set result ""
     foreach one [array names moduleArray] {
	 if !$moduleArray($one) {
   	    lappend result $one
	 }
     }	
     return $result 
}

# enableModule --
#     Enables directives for a module
# 

body apacheModuleManager::enableModule {module} {
    set moduleArray($module) 1  
}

# disableModule --
#     Disables directives for a module
# 

body apacheModuleManager::disableModule {module} {
    set moduleArray($module) 0
}

# getPropertyPagesByNodetypeList --
#          Answers backs a list with the property pages objects that are
# associated with a certain node type (mainserver, virtualhost, etc)
#
# Arguments
#     nodeType   Type of the node
#
# Returns
#     list of xuiObjects

body apacheModuleManager::getPropertyPagesByNodetypeList { nodeType } {
    set result ""
    foreach match [array names nodeArray *,[string tolower $nodeType]] {

	# Check if module enabled
         
	if $moduleArray([lindex [split $match ,] 0]) {
	    foreach page $nodeArray($match) {
	    lappend result $page
	    }
	}
    }
    return $result
}

# apacheModuleManager::addPropertyPageNodeRelationship --
#   Associates, for a certain module and nodetype a property page, the skill
# for that property page (newbie/advanced) and where to hook under that 
# property page

body apacheModuleManager::addPropertyPageNodeRelationship { module pageName \
	nodeType args } {
    array set options {-hookUnder root -skillLevel all}
    array set options $args
    if ![info exists moduleArray($module)] {
	set moduleArray($module) 1
    }
    lappend nodeArray($module,[string tolower $nodeType]) \
	    [list  $pageName $options(-skillLevel) $options(-hookUnder)]
}


# All directives that belong to that module

body apacheModuleManager::addModuleDirectivesRelationship {module directiveList} {
    foreach dir $directiveList {
	set directiveModuleRelationship([string tolower $dir]) $module
    }
}

body apacheModuleManager::isDirectiveEnabled {directive} {

    # If not exists, is unknown, thus enabled

    if ![ info exists directiveModuleRelationship($directive)] { return 1 }
    return $moduleArray($directiveModuleRelationship($directive))
}
