/* ====================================================================
 * Copyright (c) 1997-2000 Harrie Hazewinkel.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Harrie Hazewinkel."
 *
 * 4. The name of the Copyright holder must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Products derived from this software may not be called "MOD-SNMP"
 *    nor may "MOD-SNMP" appear in their names  without prior written
 *    permission of Harrie Hazewinkel.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Harrie Hazewinkel"
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * The creator of this module is harrie@mod-snmp.com
 * This file contains generated code of SMASH also created by Harrie.
 * This code is a rewrite of the prototype implementation of the
 * DESIRE/MUSIQ project. Special thanks to them.
 *
 */


 
/* General includes */
#include <sys/types.h>
#include <limits.h>
#include <sys/socket.h>
#ifdef UNIXDOMAINSOCKET
#include <sys/un.h>
#endif
 
/* Apache includes */
#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "scoreboard.h"

/* SNMP includes */
#include "asn1.h"
#include "snmp.h"
#include "agt_engine.h"
#include "snmp_community.h"
#include "www-mib.h"
#include "snmpv2-mib.h"
#include "config-mib.h"
#include "scoreboard-mib.h"
#include "mod_snmp.h"
#include "status_codes.h"

/* Is really define somewhere in a library */
extern int errno;

server_rec		*www_services; /* www_services records */
ap_generation_t		snmp_generation; /* generation of this child */

pid_t			snmpagent_pid;
int			snmp_port = SNMP_PORT;
char*			snmp_addr = "0.0.0.0";
char			*snmp_temp_dir = SNMP_AGT_BUCKET_DEFAULT_DIR;
int			snmp_module_debug_level = 0;

int			http2snmp_socket = 1;
#ifdef UNIXDOMAINSOCKET
struct sockaddr_un	http2snmp_address;
int			http2snmp_addresslength;
char			*http2snmp_name = HTTP_2_SNMP_SOCKETNAME;
#else
int			http2snmp_addresslength;
struct sockaddr_in	http2snmp_address;
#endif

/*
 * Function:
 *	void cleanup_snmp_module()
 * Description:
 *	Cleaning up function to allow restarts for the SNMP-module
 *	at (re-)start this function kills the existing SNMP-agent
 *	process and cleanup up the 'http-snmp-api'.
 */
void
cleanup_snmp_module()
{

    /* kill existing snmp-agent (if there is any) */
    if (snmpagent_pid) {
	kill(snmpagent_pid, SIGKILL);
    }
#ifdef UNXIDOMAINSOCKET 
    unlink(http2snmp_name);
#endif
}
/*
 * Function:
 *	void init_snmp_module(server_rec *s, pool *p)
 * Description:
 *	Initialisation for 'apache-servers' to use the SNMP-module. This does
 *	NOT intialise the complete SNMP-AGENT. The SNMP-AGENT allocates some
 *	memory and it is not required that the rest of the http-server this
 *	knows.
 */
void	init_snmp_module(server_rec *s, pool *p)
{
int	hostNr = 0;
int	mainport = s->port;

    /* Cleanup existing stuff for restarts */
    cleanup_snmp_module();

    snmp_generation = ap_my_generation;
 
    /* now initialise */
#ifdef UNIXDOMAINSOCKET
    http2snmp_socket = socket(AF_UNIX, SOCK_DGRAM, 0);
#else
    http2snmp_socket = socket(AF_INET, SOCK_DGRAM, 0);
#endif
    if (http2snmp_socket < 0) {
	ap_log_unixerr("open_channel", NULL,    
			"Cannot open 'http-snmp-api' socket\n", s);
    }  
    memset((char *) &http2snmp_address, 0, sizeof(http2snmp_address));
    #ifdef UNIXDOMAINSOCKET
	http2snmp_address.sun_family = AF_UNIX;
	strcpy(http2snmp_address.sun_path, http2snmp_name);
	http2snmp_address.sun_len = sizeof(http2snmp_address.sun_family) +
				strlen(http2snmp_address.sun_path) + 1;
	http2snmp_addresslength = http2snmp_address.sun_len;
    #else
	http2snmp_address.sin_family = AF_INET;
	http2snmp_address.sin_addr.s_addr = inet_addr("127.0.0.1");
	http2snmp_address.sin_port = htons(0);
	http2snmp_addresslength = sizeof(http2snmp_address);
    #endif
    if (bind(http2snmp_socket, (struct sockaddr *)&http2snmp_address,
						http2snmp_addresslength) < 0){
	ap_log_unixerr("open_channel", NULL, 
		"Cannot bind 'http-snmp-api' socket\n", s);
    }
#ifndef UNIXDOMAINSOCKET
    getsockname(http2snmp_socket, (struct sockaddr *)&http2snmp_address,
						&http2snmp_addresslength);
#endif

    /* initialise the server record with numbers */
    while (s) {
	s->vhost_index = hostNr++;
	s->port = mainport;
	s = s->next;
    }
    if (hostNr >= HARD_VIRTUAL_HOST_MAX) {
        ap_log_error_old("Number of v-hosts exceeds \"HARD_VIRTUAL_HOST_MAX\"",
                                                www_services); 
        exit(-1);
    }

}

const char *
snmp_communities (cmd_parms *cmd, void *dummy, char *arg)
{
    if ( strlen(arg) > MAX_COMMUNITY_LEN ) 
		return "Community string too long";
    /* A little over the top, but is more easy to 
     * use the same of the SNMP-lib.
     */
    return set_community(NULL, arg);
}

int	snmp_section = 0;
char	*start_snmp_config_magic = "<SNMP> out of place";
char	*end_snmp_config_magic = "</SNMP> out of place";
char	*outside_snmp_config_magic = "Not in SNMP section";

const char *
start_snmp_section (cmd_parms *cmd, void *dummy, char *argu)
{
    char *endp;
    char *colon;

    if (snmp_section) return "SNMP does not nest";
    snmp_section = 1;
    endp = strrchr(argu, '>');
    if (endp) *endp = '\0';
    colon = strrchr(argu, ':');
    if (colon) {
        *colon = '\0';
        snmp_addr =  argu;
        snmp_port = atoi(++colon);
    } else {
        snmp_port = atoi (argu);
    }
    return NULL;
}

const char *
end_snmp_section(cmd_parms *cmd, void *dummy)
{
    if (snmp_section) {
	snmp_section = 0;
        return NULL;
    }
    return end_snmp_config_magic;
}

const char *
snmp_buckets(cmd_parms *cmd, void *dummy, char *arg)
{
    if (snmp_section) {
	snmp_temp_dir = ap_pstrdup(cmd->pool, arg);
	return NULL;
    }
    return "Not specified in SNMP-section";
}

#ifdef UNIXDOMAINSOCKET
const char *
init_http2snmp_socketname(cmd_parms *cmd, void *dummy, char *arg)
{
    pool *p = cmd->pool;
    int i = strlen(arg);

    if (i > MAX_OCTSTR_LEN) {
	i = MAX_OCTSTR_LEN;
    }
    http2snmp_name = (char *)ap_pcalloc (p, i+1);
    if (http2snmp_name) {
	strncpy(http2snmp_name, arg, i);
    }
    return NULL;
}
#endif

char *bucketDir;

/*
 * Function:
 *	int     init_wwwProtocolStatistics(server_rec *s, pool *p)
 * Description:
 *	Initialisation of the www protocol statistics (protocol and document).
 *	Since it uses a lot of allocated memory only done within
 *	SNMP-agent. All values are set to the MIB defined default!!
 */
int	init_wwwProtocolStatistics(server_rec *s, pool *p)
{
WwwStats		*wwwStats;


    while (s) {
	wwwStats = (WwwStats *)ap_pcalloc (p, sizeof(WwwStats));
	if (wwwStats) {
	    s->snmp_www_stats = wwwStats;

	    wwwStats->summaryInRequests = 0;
	    wwwStats->summaryOutRequests = 0;
	    wwwStats->summaryInResponses = 0;
	    wwwStats->summaryOutResponses = 0;
	    wwwStats->summaryInLowBytes = 0;
	    wwwStats->summaryOutLowBytes = 0;
#ifdef WWW_DOCUMENTS_GROUP
	    wwwStats->docCtrlLastNSize = DEFAULT_LASTNSIZE;
	    wwwStats->docCtrlLastNLock = 0;
	    wwwStats->docCtrlBuckets = DEFAULT_BUCKETS;
	    wwwStats->docCtrlBucketTimeInterval = 9000; 
	    wwwStats->docCtrlTopNSize = DEFAULT_TOPNSIZE;
	    wwwStats->docLastNIndex = 0;
	    wwwStats->docBucketIndex = 0;
	    wwwStats->bucketName = ap_pstrcat(p, 
			ap_server_root_relative(p, snmp_temp_dir),
					"/bucket.", s->server_hostname, NULL);
	    wwwStats->bucket = ap_bucket_open(wwwStats->bucketName);
	    if (!wwwStats->bucket) {
		ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, s,
			"SNMP:ap_bucket_open: %s '%s'", ap_bucket_strerror(),
			wwwStats->bucketName);
	    }
#endif /* WWW_DOCUMENTS_GROUP */
	}
	s = s->next;
    }
    return(OK);
}

/*
 * Function:
 *	int	 http_log_lastDocument(request_rec *r)
 * Description:
 * This function fills the document log structure at the http-side
 * of the 'http-to-snmp-api' and sends (datagram) it to the snmp-side.
 */
int
http_log_Document(request_rec *r)
{
    int			i;
    ApiMsg        document;

    document.serviceIndex = r->server->vhost_index;
    document.requestTime = r->request_time;
    document.bytesIn = r->bytes_recv;
    if (!r->sent_bodyct) {
	document.bytesOut = 0;
    } else {
	ap_bgetopt(r->connection->client, BO_BYTECT, &document.bytesOut);
    }
    strncpy(document.docName, r->uri, sizeof(document.docName));
    sprintf(document.statusMsg, "host '%d'", r->server->vhost_index);
    document.requestInType = -1;
    for (i=0 ; i<(METHODS-1) ; i++) {
	if(!strcmp(r->method, requestTypes[i])) {
		document.requestInType = i;
		break;
	}
    }
    i = 0;
    while (responseTypes[i] > 0) {
	if(responseTypes[i] == r->status) {
		document.responseOutType = i;
	    break;
	}
	i++;
    }
    if (0 >= sendto(http2snmp_socket, (char *)&document, sizeof(document), 0,
		(struct sockaddr *)&http2snmp_address, sizeof(http2snmp_address))) {
	perror("sendto");
    }
    return(OK);
}


int
snmp_log_lastDocument(int socket)
{
ApiMsg		lastDoc;
server_rec	*s;
WwwStats	*wwwStats;
#ifdef WWW_DOCUMENTS_GROUP
int		lastNIndex;
WwwDocTopN	documentData;
WwwDocTopN	*ptr_documentData;
#endif /* WWW_DOCUMENTS_GROUP */

    if (read(socket, &lastDoc, sizeof(lastDoc)) != sizeof(lastDoc)) {
	return(!OK);
    }

    /* Search for the correct ServiceIndex */
    for (s = www_services ; s ; s = s->next) {
	if (s->vhost_index == lastDoc.serviceIndex) {
	    break;
	}
    }

    if (s) {
	wwwStats = (WwwStats *)s->snmp_www_stats;
	if (wwwStats) {
	    wwwStats->summaryInRequests++;
	    wwwStats->summaryOutResponses++;
	    wwwStats->summaryInLowBytes += lastDoc.bytesIn;
	    wwwStats->summaryOutLowBytes += lastDoc.bytesOut;
#ifdef WWW_REQUEST_IN_GROUP
	    if (lastDoc.requestInType >= 0){
		wwwStats->requestInEntries[ lastDoc.requestInType ].count++;
		wwwStats->requestInEntries[ lastDoc.requestInType ].bytes +=
					lastDoc.bytesIn;
		wwwStats->requestInEntries[ lastDoc.requestInType ].lastTime =
					lastDoc.requestTime;
	    }
#endif /* WWW_REQUEST_IN_GROUP */
#ifdef WWW_RESPONSE_OUT_GROUP
	    wwwStats->responseOutEntries[ lastDoc.responseOutType ].count++;
	    wwwStats->responseOutEntries[ lastDoc.responseOutType ].bytes +=
					lastDoc.bytesOut;
	    wwwStats->responseOutEntries[ lastDoc.responseOutType ].lastTime =
					lastDoc.requestTime;
#endif /* WWW_RESPONSE_OUT_GROUP */
#ifdef WWW_DOCUMENTS_GROUP
	    if (wwwStats->docLastNTable) {
		wwwStats->docLastNIndex++;
		lastNIndex = wwwStats->docLastNIndex % MAX_LASTNSIZE;
		strncpy(wwwStats->docLastNTable[ lastNIndex ].docName,
					lastDoc.docName, MAX_DOCNAME);
		wwwStats->docLastNTable[ lastNIndex ].requestTime =
					lastDoc.requestTime;
		wwwStats->docLastNTable[ lastNIndex ].requestType =
					lastDoc.requestInType;
		wwwStats->docLastNTable[ lastNIndex ].responseType =
					lastDoc.responseOutType;
		strncpy(wwwStats->docLastNTable[ lastNIndex ].statusMsg,
					lastDoc.statusMsg, MAX_STATMSG);
		wwwStats->docLastNTable[ lastNIndex ].docBytes =
					lastDoc.bytesOut;
	    }
	    ptr_documentData = ap_bucket_fetch(wwwStats->bucket, lastDoc.docName);
	    if (ptr_documentData) {
		/* found in bucket; update document document data */

		ptr_documentData->accesses++;
		ptr_documentData->bytes += lastDoc.bytesOut;
		ptr_documentData->lastResponseType = lastDoc.responseOutType;
		ptr_documentData->lastAccess = lastDoc.requestTime;
		if (ap_bucket_store(wwwStats->bucket, lastDoc.docName,
					ptr_documentData, RECORD_REPLACE)) {
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, www_services,
					"ap_bucket_store '%s'", ap_bucket_strerror());
		}
	     } else {
		/* not found in bucket; insert document data */
		strcpy(documentData.name, lastDoc.docName);
		documentData.accesses = 1;
		documentData.bytes = lastDoc.bytesOut;
		documentData.lastResponseType = lastDoc.responseOutType;
		documentData.lastAccess = lastDoc.requestTime;
		if (ap_bucket_store(wwwStats->bucket, lastDoc.docName, &documentData,
		    					RECORD_INSERT)) {
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, www_services,
					"ap_bucket_store '%s'", ap_bucket_strerror());
		}

	    }
#endif /* WWW_DOCUMENTS_GROUP */
	}
    }
    return(OK);
}

#ifdef WWW_DOCUMENTS_GROUP
void
insert_in_accessTopNTable(WwwDocTopN *doc, WwwDocTopN *accessTopNTable, int size)
{
int	i;

    size = size-2; /* We correct this for the array */
    for ( i=0 ; i<size ; i++) {
	if (accessTopNTable[i].accesses <= doc->accesses) {
	    while (i <= size) {
		memcpy(&(accessTopNTable[size+1]), &(accessTopNTable[size]),
							sizeof(WwwDocTopN));
		size--;
	    }
	    memcpy(&(accessTopNTable[i]), doc, sizeof(WwwDocTopN));
	    for (i=0; i<25 ; i++) {
	    }
	    return;
	}
    }
    if (accessTopNTable[i].accesses <= doc->accesses) {
	memcpy(&(accessTopNTable[i]), doc, sizeof(WwwDocTopN));
    }
}

void
insert_in_bytesTopNTable(WwwDocTopN *doc, WwwDocTopN *bytesTopNTable, int size)
{
int     i;

    size = size-2; /* We correct this for the array */
    for ( i=0 ; i<size ; i++) {
	if (bytesTopNTable[i].bytes <= doc->bytes) {
	    while (i <= size) {
		memcpy(&(bytesTopNTable[size+1]), &(bytesTopNTable[size]),
							sizeof(WwwDocTopN));
		size--;
	    }
	    memcpy(&(bytesTopNTable[i]), doc, sizeof(WwwDocTopN));
	    for (i = 0 ; i<25 ; i++) {
	    }
	    return;
	}
    }
    if (bytesTopNTable[i].bytes <= doc->bytes) {
	memcpy(&(bytesTopNTable[i]), doc, sizeof(WwwDocTopN));
    }
}

void
set_alarm_bucketsTimeInterval(server_rec *service)
{   
WwwStats	*wwwStats = service->snmp_www_stats;
    
    if (wwwStats) {
	if (wwwStats->docCtrlBucketTimeInterval > 0) {
	    alarm(wwwStats->docCtrlBucketTimeInterval/100);
	}       
    }           
}               

void
compute_bucket_table(WwwStats *wwwStats)
{
WwwDocBucket	*docBucket;
WwwDocTopN	*documentData;

    docBucket = &(wwwStats->docBucketTable[++wwwStats->docBucketIndex % 
						MAX_BUCKETS ]);;
    if (docBucket) {
	docBucket->bucketCreateTime = time(NULL);
	docBucket->accesses = 0;
	docBucket->documents = 0;
	docBucket->bytes = 0;
	if (docBucket->accessTopNTable) {
	    memset(docBucket->accessTopNTable, 0,
		(sizeof(WwwDocTopN) * MAX_LASTNSIZE));
	}
	if (docBucket->bytesTopNTable) {
	    memset(docBucket->bytesTopNTable, 0,
		(sizeof(WwwDocTopN) * MAX_LASTNSIZE));
	}
	/* We now go through the complete bucket and sort the documentData */
	documentData = ap_bucket_seqfetch(wwwStats->bucket, FIRST_RECORD);
	while (documentData) {
	    insert_in_accessTopNTable(documentData, docBucket->accessTopNTable,
					wwwStats->docCtrlTopNSize);
	    insert_in_bytesTopNTable(documentData, docBucket->bytesTopNTable,
					wwwStats->docCtrlTopNSize);
	    docBucket->accesses += documentData->accesses;
	    docBucket->documents++;
	    docBucket->bytes += documentData->bytes;
	    documentData = ap_bucket_seqfetch(wwwStats->bucket, NEXT_RECORD);
	}
    }
}

/* This function walks through to compute all bucket related tables.
 * It does them always together, but it should be possible to do this
 * independent.
 */
void
compute_buckets(int signal)
{
server_rec	*s = www_services;
WwwStats	*wwwStats;

    set_alarm_bucketsTimeInterval(s); 
    while (s) {
	wwwStats = s->snmp_www_stats;
	if (wwwStats) {
	    compute_bucket_table(wwwStats);
	    ap_bucket_close(wwwStats->bucket);
	    wwwStats->bucket = ap_bucket_open(wwwStats->bucketName);
	    if (!wwwStats->bucket) {
		ap_log_unixerr("ap_bucket_open", NULL, ap_bucket_strerror(), s);
	    }
	}
	s = s->next;
    }
}

void
init_alarm_bucketsTimeInterval()
{
#ifndef NO_USE_SIGACTION
struct sigaction snmp_sig;

    sigemptyset(&snmp_sig.sa_mask);
    snmp_sig.sa_flags = 0;
    snmp_sig.sa_handler = (void (*)())compute_buckets;
    if (sigaction (SIGALRM, &snmp_sig, NULL) < 0)
	ap_log_unixerr ("sigaction(SIGALRM)", NULL, NULL, www_services);
    sigaddset (&snmp_sig.sa_mask, SIGALRM);
#else
    signal (SIGALRM, (void (*)())compute_buckets);
#endif
}
#endif /* WWW_DOCUMENTS_GROUP */

void
snmp_agt_main(server_rec *s, pool *p, pool *ptemp)
{
int     count;
int     snmp_socket;
int     numfds;
fd_set  fdset;

    snmpagent_pid = fork();
    if (snmpagent_pid == -1) { /* Fork error */
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, s,
					"SNMP: unable to fork agent");
    } else if (snmpagent_pid > 0) { 
	return;
    }

    /* Open the network */
    snmp_socket = snmp_open_connection(snmp_addr, snmp_port);
    if (snmp_socket == 0) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, s,
						"SNMP: cannot open socket");
	exit(0);
    }
    ensure_communities();

    init_SNMPv2_MIB();
    init_WWW_MIB();
    init_APACHE_CONFIG_MIB();
    init_APACHE_SCOREBOARD_MIB();
    init_wwwServiceOperStatus();
    init_wwwProtocolStatistics(s, p);
    www_services = s;

#ifdef WWW_DOCUMENTS_GROUP
    init_alarm_bucketsTimeInterval();
    set_alarm_bucketsTimeInterval(s);
#endif

    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, s,
                    "SNMP: agent started and listens on port '%s:%d'", snmp_addr, snmp_port);

    /* Listen to the network */
    while(1){
        numfds = 0;
        FD_ZERO(&fdset);
	FD_SET(snmp_socket, &fdset);
        FD_SET(http2snmp_socket, &fdset);
        if (snmp_socket > http2snmp_socket) {
	    numfds = snmp_socket+1;
	} else {
	    numfds = http2snmp_socket+1;
	}
        count = select(numfds, &fdset, 0, 0, 0);
        if (count > 0){
            if (FD_ISSET(snmp_socket, &fdset)) {
                snmp_process_message(snmp_socket);
            } else if (FD_ISSET(http2snmp_socket, &fdset)) {
		snmp_log_lastDocument(http2snmp_socket);
	    }
        } else switch(count){
            case 0:
                break;
            case -1:
                if (errno == EINTR) {
		    continue;
                } else {
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, s,
				"SNMP: select error '%s'\n", strerror(errno));
                }
            default:
		ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, s,
                		"SNMP: select returned %d\n", count);
        }
    }
}


command_rec config_cmds_snmp_module[] = {
{ "<SNMP", start_snmp_section, NULL, RSRC_CONF, TAKE1, "start of SNMP config section"},
{ "</SNMP>", end_snmp_section, NULL, RSRC_CONF, NO_ARGS, "end of SNMP config section"},
{ "SNMPcommunity", snmp_communities, NULL, RSRC_CONF, ITERATE, "error with communities"},
{ "SNMPbuckets", snmp_buckets, NULL, RSRC_CONF, RAW_ARGS, "error for buckets directory"},
{ "sysDescr", init_sysDescr, NULL, RSRC_CONF, RAW_ARGS, "incorrect sysDescr"},
{ "sysContact", init_sysContact, NULL, RSRC_CONF, RAW_ARGS, "incorrect sysContact"},
{ "sysLocation", init_sysLocation, NULL, RSRC_CONF, RAW_ARGS, "incorrect sysLocation"},
{ NULL }
};

module MODULE_VAR_EXPORT snmp_agt_module = {
    STANDARD_MODULE_STUFF,
    init_snmp_module,		/* initializer */
    NULL,			/* create per-directory config structure */
    NULL,			/* merge per-directory config structure */
    NULL,			/* create per-server config structure */
    NULL,			/* merge per-server config structure */
    config_cmds_snmp_module,	/* command table */
    NULL,			/* handlers */
    NULL,			/* translate handlers */
    NULL,			/* check_user_id */
    NULL,			/* check auth */
    NULL,			/* check access */
    NULL,			/* type_checker */
    NULL,			/* pre-run fixups */
    http_log_Document,	 	/* logger */
    NULL,			/* header parser */
    NULL,			/* child_init */
    NULL,			/* child_exit */
    NULL			/* post read-request */
};


